import bpy
import io
import json
from mathutils import Quaternion, Matrix, Vector, Euler, Color
import logging
import math
import traceback
import sys
from s4studio.helpers import FNV32
from s4studio.model.geometry import Vertex, BoundingBox

AXIS_FIXER = Quaternion((0.7071067690849304, 0.7071067690849304, 0.0, 0.0))


def eprint(*args, **kwargs):
    print(*args, file=sys.stderr, **kwargs)

# load texture
def load_material(map_name, map_resource, is_cas=True):
    material = bpy.data.materials.new(map_name)
    texture_index = 0

    material.specular_color = Color([.1, .1, .1])

    bt = 'BaseTexture_BaseTexture'
    # Add base texture
    if is_cas and bt in bpy.data.textures:
        material.texture_slots.create(texture_index)
        texture_slot = material.texture_slots[texture_index]
        texture_index += 1
        texture = bpy.data.textures.new(name=map_name + '_' + material.name, type='IMAGE')
        texture_slot.texture = texture
        texture_slot.texture_coords = 'UV'
        texture.type = 'IMAGE'
        texture.image = bpy.data.textures[bt].image
        texture_index += 1

    bpy.context.scene.render.image_settings.file_format = 'PNG'
    bpy.context.scene.render.image_settings.color_mode = 'RGBA'
    bpy.ops.image.open(filepath=map_resource)
    img = bpy.data.images.new(map_name, 512, 512)
    img.source = 'FILE'
    img.filepath = map_resource
    img.save_render(map_resource)
    img.filepath = map_resource

    material.texture_slots.create(texture_index)
    texture_slot = material.texture_slots[texture_index]
    texture_index += 1
    texture = bpy.data.textures.new(name=map_name + '_' + material.name, type='IMAGE')
    texture_slot.texture = texture
    texture_slot.texture_coords = 'UV'

    if map_name in ('DiffuseMap', 'Multiplier'):
        texture_slot.use_stencil = True
    elif map_name == 'Mask':
        texture_slot.use = False
        texture_slot.blend_type = 'OVERLAY'
        texture_slot.use_rgb_to_intensity = True
    elif map_name in ('Specular', 'SpecularMap', 'Clothing Specular'):
        texture_slot.use_map_specular = True
        texture_slot.use_map_color_spec = True
        texture_slot.use_map_hardness = True
        texture_slot.use_map_color_diffuse = False
    elif map_name == 'NormalMap':
        texture_slot.use_map_normal = True
        texture_slot.normal_factor = 0.01
        texture_slot.use_map_color_diffuse = False
    texture.type = 'IMAGE'
    texture.image = img
    return material


def s3_4x3_to_Matrix(s3_matrix):
    """
    Arranges a sequence of floats into a mathutils.Matrix
    """
    args = [tuple(m) for m in s3_matrix]
    args.append((0.0, 0.0, 0.0, 1.0))
    return Matrix(args)


def swizzle_uv(uv):
    return uv[0], 1 - uv[1]


def unswizzle_uv(uv):
    return [uv[0], 1 - uv[1]]


def swizzle_v3(v3):
    return Vector([v3[0], v3[1], v3[2]])


def quat_wxyz(quaternion):
    """
    Swap xyzw (order used by The Sims 3) to wxyz(order used by Blender).
    """
    return quaternion[3], quaternion[0], quaternion[1], quaternion[2]


def argb_rgb(argb):
    return argb[1:3]


def quat_xyzw(quaternion):
    return [quaternion[1], quaternion[2], quaternion[3], quaternion[0]]


def invalid_face(face):
    if not face: return True
    t = []
    for f in face:
        if f not in t:
            t.append(f)
    return len(t) != len(face)


def create_marker_node(name, rotate=False):
    set_context('OBJECT')
    bpy.ops.object.add(type='EMPTY')
    marker = bpy.context.active_object
    marker.name = name
    marker.show_x_ray = True
    marker.empty_draw_size = .1
    marker.empty_draw_type = 'CUBE'
    if rotate:
        marker.rotation_euler = Euler([math.pi / 2, 0, 0])
        # bpy.ops.transform.rotate(value=(math.pi / 2), axis=(1, 0, 0))
    return marker


def apply_all_modifiers(ob):
    set_context('OBJECT',ob)
    bpy.ops.object.convert(target='MESH')

def set_context(mode=None, select=None):
    current_active = bpy.context.scene.objects.active
    current_mode = bpy.context.object.mode if bpy.context.object else None
    for obj in bpy.data.objects:
        obj.select = obj == select
    if current_active == select and mode == current_mode:
        return
    if current_active != select:
        if current_active:
            if current_mode != 'OBJECT':
                current_active.hide = False
                current_active.hide_select = False
                bpy.ops.object.mode_set(mode='OBJECT')
        bpy.context.scene.objects.active = select
    if bpy.context.object and bpy.context.object.mode != mode:
        bpy.ops.object.mode_set(mode=mode)


def approximate_vector(v, precision=2):
    return str([round(x, precision) for x in v])
    pass


def equals_float_array(a, b, precision=2):
    assert len(a) == len(b)
    for i in range(len(a)):
        if round(a[i], precision) != round(b[i], precision):
            return False
    return True


def equals_vector(a, b, precision=.25):
    v1 = Vector(a)
    v2 = Vector(b)
    v3 = v1 - v2
    l = math.fabs(v3.length)
    return l < precision


blend_index_map = [0, 1, 2, 3]


class SimMeshData(object):
    def __init__(self):
        self.vertices = []
        self.indices = []
        self.bones = []
        self.bone_names = []
        self.bounds = BoundingBox()
        self.bounds.init_values()


def collect_mesh_data(mesh_object):
    bvmajor, bvminor, bvrevision = bpy.app.version

    mesh_data = mesh_object.data
    vertex_group_map = {}
    vertex_groups = []
    bone_names = []
    dta = SimMeshData()
    if len(mesh_data.loops) == 0:
        return dta
    set_context('OBJECT', mesh_object)
    set_context('EDIT', mesh_object)
    bpy.ops.mesh.reveal()
    bpy.ops.mesh.select_all(action='SELECT')
    bpy.ops.mesh.quads_convert_to_tris()
    bpy.ops.mesh.select_all(action='DESELECT')
    set_context('OBJECT', mesh_object)
    bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
    bpy.ops.transform.rotate(value=math.pi / 2.0, axis=(-1, 0, 0))
    bpy.ops.object.transform_apply(rotation=True)
    has_uv = len(mesh_data.uv_layers) > 0
    if has_uv:
        mesh_data.calc_tangents()
    # Establish bone groups
    for vertex_group_index in range(len(mesh_object.vertex_groups)):
        vertex_group = mesh_object.vertex_groups[vertex_group_index]
        hash = FNV32.hash(vertex_group.name)
        if not hash in vertex_groups:
            vertex_groups.append(hash)
        if not vertex_group.name in bone_names:
            bone_names.append(vertex_group.name)
        vertex_group_map[vertex_group.name] = vertex_groups.index(hash)
    pass

    sim_vertices = []
    indices = []

    # Vertex Index matched to a list of Sim Vertices split by UV coordinates
    vertex_map = {}

    # Enumerate blender loops to split vertices if their loop is different
    for loop_index, loop in enumerate(mesh_data.loops):
        face_point_index = loop_index % 3
        face_index = int((loop_index - face_point_index) / 3)

        # Add a triangle
        if face_point_index == 0:
            indices.append([])

        # Initialize vertex map for this index
        if not loop.vertex_index in vertex_map:
            vertex_map[loop.vertex_index] = []

        # Final vertex for face
        sim_vertex = None

        # Collect UV coordinates for all layers
        loop_uv = []
        if has_uv:
            for uv_layer_index, uv_layer in enumerate(mesh_data.uv_layers):
                uv = uv_layer.data[loop_index].uv
                loop_uv.append(swizzle_uv(uv))

        # Collect tangent vector for loop
        loop_tangent = [1, 0, 0]
        if has_uv:
            loop_tangent = [loop.tangent.x, loop.tangent.y, loop.tangent.z, loop.bitangent_sign]

        use_custom_normals = False

        if bvmajor >= 2 and bvminor >= 74 and use_custom_normals:
            loop_normal = [loop.normal.x, loop.normal.y, loop.normal.z]
        else:
            vertex_normal = mesh_data.vertices[loop.vertex_index].normal
            loop_normal = [vertex_normal.x, vertex_normal.y, vertex_normal.z]

        # Check for existing matching vertex
        for v in vertex_map[loop.vertex_index]:
            assert isinstance(v, Vertex)
            # Compare loop's UV to existing vertex
            uv_equal = True
            for i in range(len(v.uv)):
                if not equals_float_array(loop_uv[i], v.uv[i], 4):
                    uv_equal = False
                    break
            # Compare loop's normal to existing vertex
            normal_equal = False
            tangent_equal = not has_uv
            if bvmajor >= 2 and bvminor >= 74 and use_custom_normals:
                normal_equal = equals_vector(loop_normal, v.normal)
                if has_uv:
                    tangent_equal = equals_vector(loop_tangent, v.tangent)
            else:
                normal_equal = True
                tangent_equal = True

            if uv_equal and normal_equal:  # and tangent_equal:
                sim_vertex = v
                break
        # Create a new vertex if no matches are found
        if not sim_vertex:
            # Collect blender vertex
            vertex = mesh_data.vertices[loop.vertex_index]
            sim_vertex = Vertex()
            sim_vertex.uv = []
            # Set position
            sim_vertex.position = [vertex.co.x, vertex.co.y, vertex.co.z]
            dta.bounds.add(Vector(sim_vertex.position))
            # Set normal vector
            sim_vertex.normal = loop_normal
            # Set bone weights
            blend_index = [0] * 4
            weight_list = [1.0, 0, 0, 0]
            for i in range(min(len(vertex.groups), 4)):
                bone_idx = vertex_group_map[mesh_object.vertex_groups[int(vertex.groups[i].group)].name]
                if i == 0:
                    blend_index = [bone_idx] * 4
                weight_list[blend_index_map[i]] = min(vertex.groups[i].weight, 1.0)
                blend_index[i] = bone_idx
            sim_vertex.blend_indices = blend_index
            sim_vertex.blend_weights = weight_list

            # Set vertex color
            if len(mesh_data.vertex_colors):
                blender_color = mesh_data.vertex_colors[0].data[loop_index].color[:]
                sims_color = [c for c in blender_color]
                sims_color.append(0)
                sim_vertex.colour = sims_color

            if has_uv:
                # Set loop data
                sim_vertex.uv = loop_uv
                sim_vertex.tangent = loop_tangent

            # Add new vertex to the main list
            sim_vertices.append(sim_vertex)
            vertex_map[loop.vertex_index].append(sim_vertex)

        indices[face_index].append(sim_vertices.index(sim_vertex))

    split_count = 0
    for i in vertex_map.keys():
        verts = vertex_map[i]
        split_count += 1
    print('%s vertices split.' % split_count)

    final_indices = []
    for face in indices:
        a = [face[2], face[0], face[1]]
        for i in a:
            final_indices.append(i)

    dta.indices = final_indices
    dta.vertices = sim_vertices
    dta.bones = vertex_groups
    dta.bone_names = bone_names
    return dta


class Sims4StudioException(Exception):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
